<?php
/**
 * My Jobs - Job Portal
 * GHAMECC Job Portal
 */

require_once '../config/database.php';
require_once '../includes/auth.php';

// Check if user is logged in
if (!isLoggedIn()) {
    header('Location: login.php');
    exit();
}

$userId = $_SESSION['user_id'];

// Get user's posted jobs
$jobs = fetchRows("
    SELECT j.*, jc.name as category_name,
           (SELECT COUNT(*) FROM job_applications ja WHERE ja.job_id = j.id) as application_count
    FROM jobs j
    LEFT JOIN job_categories jc ON j.category_id = jc.id
    WHERE j.posted_by_user_id = ?
    ORDER BY j.created_at DESC
", [$userId]);

$pageTitle = "My Jobs";
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?> - GHAMECC Job Portal</title>
    
    <?php include '../includes/favicon.php'; ?>
    
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .job-card {
            transition: transform 0.2s;
        }
        .job-card:hover {
            transform: translateY(-2px);
        }
        .status-badge {
            font-size: 0.8rem;
        }
        .status-open { background-color: #28a745; }
        .status-closed { background-color: #6c757d; }
        .status-filled { background-color: #17a2b8; }
        .approval-pending { background-color: #ffc107; }
        .approval-approved { background-color: #28a745; }
        .approval-rejected { background-color: #dc3545; }
    </style>
</head>
<body>
    <?php include 'includes/job-portal-navbar.php'; ?>

    <div class="container mt-4">
        <div class="row">
            <div class="col-12">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2><i class="fas fa-briefcase me-2"></i><?php echo $pageTitle; ?></h2>
                    <a href="post-job.php" class="btn btn-primary">
                        <i class="fas fa-plus me-2"></i>Post New Job
                    </a>
                </div>

                <?php if (empty($jobs)): ?>
                    <div class="text-center py-5">
                        <i class="fas fa-plus-circle fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No Jobs Posted Yet</h4>
                        <p class="text-muted">Start posting job opportunities to connect with talented members!</p>
                        <a href="post-job.php" class="btn btn-primary">
                            <i class="fas fa-plus me-2"></i>Post Your First Job
                        </a>
                    </div>
                <?php else: ?>
                    <div class="row">
                        <?php foreach ($jobs as $job): ?>
                            <div class="col-md-6 col-lg-4 mb-4">
                                <div class="card job-card h-100">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between align-items-start mb-2">
                                            <h5 class="card-title"><?php echo htmlspecialchars($job['title']); ?></h5>
                                            <div class="d-flex flex-column align-items-end">
                                                <span class="badge status-badge status-<?php echo $job['status']; ?> mb-1">
                                                    <?php echo ucfirst($job['status']); ?>
                                                </span>
                                                <?php if (isset($job['approval_status'])): ?>
                                                    <span class="badge status-badge approval-<?php echo $job['approval_status']; ?>">
                                                        <?php echo ucfirst($job['approval_status']); ?>
                                                    </span>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                        
                                        <h6 class="card-subtitle mb-2 text-muted">
                                            <i class="fas fa-building me-1"></i>
                                            <?php echo htmlspecialchars($job['company_name']); ?>
                                        </h6>
                                        
                                        <div class="mb-2">
                                            <small class="text-muted">
                                                <i class="fas fa-map-marker-alt me-1"></i>
                                                <?php echo htmlspecialchars($job['location']); ?>
                                            </small>
                                        </div>
                                        
                                        <?php if ($job['category_name']): ?>
                                            <div class="mb-2">
                                                <span class="badge bg-secondary">
                                                    <?php echo htmlspecialchars($job['category_name']); ?>
                                                </span>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <?php if ($job['salary_range']): ?>
                                            <div class="mb-2">
                                                <small class="text-success">
                                                    <i class="fas fa-money-bill-wave me-1"></i>
                                                    <?php echo htmlspecialchars($job['salary_range']); ?>
                                                </small>
                                            </div>
                                        <?php endif; ?>
                                        
                                        <div class="mb-3">
                                            <small class="text-muted">
                                                Posted: <?php echo date('M j, Y', strtotime($job['created_at'])); ?>
                                            </small>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <div class="d-flex justify-content-between align-items-center">
                                                <small class="text-info">
                                                    <i class="fas fa-users me-1"></i>
                                                    <?php echo $job['application_count']; ?> Applications
                                                </small>
                                                <small class="text-muted">
                                                    Deadline: <?php echo date('M j', strtotime($job['deadline'])); ?>
                                                </small>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="card-footer bg-transparent">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <div class="btn-group" role="group">
                                                <a href="job-details.php?id=<?php echo $job['id']; ?>" class="btn btn-sm btn-outline-primary">
                                                    <i class="fas fa-eye me-1"></i>View
                                                </a>
                                                <?php if ($job['application_count'] > 0): ?>
                                                    <a href="job-applications.php?job_id=<?php echo $job['id']; ?>" class="btn btn-sm btn-outline-info">
                                                        <i class="fas fa-users me-1"></i>Applications
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                            <div class="dropdown">
                                                <button class="btn btn-sm btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                                    <i class="fas fa-cog"></i>
                                                </button>
                                                <ul class="dropdown-menu">
                                                    <li><a class="dropdown-item" href="edit-job.php?id=<?php echo $job['id']; ?>">
                                                        <i class="fas fa-edit me-2"></i>Edit
                                                    </a></li>
                                                    <?php if ($job['status'] === 'open'): ?>
                                                        <li><a class="dropdown-item" href="#" onclick="updateJobStatus(<?php echo $job['id']; ?>, 'closed')">
                                                            <i class="fas fa-pause me-2"></i>Close
                                                        </a></li>
                                                    <?php else: ?>
                                                        <li><a class="dropdown-item" href="#" onclick="updateJobStatus(<?php echo $job['id']; ?>, 'open')">
                                                            <i class="fas fa-play me-2"></i>Reopen
                                                        </a></li>
                                                    <?php endif; ?>
                                                    <li><hr class="dropdown-divider"></li>
                                                    <li><a class="dropdown-item text-danger" href="#" onclick="deleteJob(<?php echo $job['id']; ?>)">
                                                        <i class="fas fa-trash me-2"></i>Delete
                                                    </a></li>
                                                </ul>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <div class="mt-4 text-center">
                        <p class="text-muted">
                            Total Jobs Posted: <strong><?php echo count($jobs); ?></strong>
                        </p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        function updateJobStatus(jobId, status) {
            if (confirm('Are you sure you want to ' + status + ' this job?')) {
                fetch('ajax/update_job_status.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        job_id: jobId,
                        status: status
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error updating job status');
                    }
                });
            }
        }
        
        function deleteJob(jobId) {
            if (confirm('Are you sure you want to delete this job? This action cannot be undone.')) {
                fetch('ajax/delete_job.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        job_id: jobId
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        location.reload();
                    } else {
                        alert('Error deleting job');
                    }
                });
            }
        }
    </script>
</body>
</html>
