<?php
/**
 * Job Portal - Landing Page
 * GHAMECC Choir Management Platform
 * 
 * This is the main landing page for the GHAMECC Job Portal.
 */

// Include database connection and functions
require_once '../config/database.php';
require_once '../includes/functions.php';
require_once '../includes/auth.php';

// Get latest jobs
$latestJobs = fetchRows("SELECT j.*, c.name as category_name 
                        FROM jobs j 
                        LEFT JOIN job_categories c ON j.category_id = c.id 
                        WHERE j.status = 'open' AND j.approval_status = 'approved'
                        ORDER BY j.created_at DESC 
                        LIMIT 6");

// Get featured profiles
$featuredProfiles = fetchRows("SELECT mp.*, m.first_name, m.last_name, 
                              (SELECT AVG(rating) FROM member_ratings WHERE rated_id = mp.member_id) as avg_rating,
                              (SELECT COUNT(*) FROM member_ratings WHERE rated_id = mp.member_id) as rating_count
                              FROM member_profiles mp
                              JOIN members m ON mp.member_id = m.id
                              ORDER BY avg_rating DESC, rating_count DESC
                              LIMIT 4");

// Get job categories with counts
$categories = fetchRows("SELECT c.*, COUNT(j.id) as job_count 
                        FROM job_categories c
                        LEFT JOIN jobs j ON c.id = j.category_id AND j.status = 'open' AND j.approval_status = 'approved'
                        GROUP BY c.id
                        ORDER BY job_count DESC
                        LIMIT 8");

// Page title
$pageTitle = "GHAMECC Job Portal - Find Jobs & Talent";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?></title>
    
    <?php include '../includes/favicon.php'; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <!-- Custom CSS -->
    <style>
        .hero-section {
            background: linear-gradient(rgba(0, 0, 0, 0.7), rgba(0, 0, 0, 0.7)), url('../assets/images/job-portal-bg.jpg');
            background-size: cover;
            background-position: center;
            color: white;
            padding: 100px 0;
            margin-bottom: 40px;
        }
        .card-job {
            transition: transform 0.3s;
            height: 100%;
        }
        .card-job:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        .profile-card {
            transition: transform 0.3s;
            height: 100%;
        }
        .profile-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        .category-card {
            transition: all 0.3s;
            border-radius: 10px;
            overflow: hidden;
            height: 100%;
        }
        .category-card:hover {
            background-color: #f8f9fa;
            transform: scale(1.03);
        }
        .star-rating {
            color: #ffc107;
        }
        .section-heading {
            position: relative;
            padding-bottom: 15px;
            margin-bottom: 30px;
        }
        .section-heading:after {
            content: '';
            position: absolute;
            bottom: 0;
            left: 0;
            width: 50px;
            height: 3px;
            background-color: #0d6efd;
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <?php include 'includes/job-portal-navbar.php'; ?>
    
    <!-- Hero Section -->
    <section class="hero-section">
        <div class="container text-center">
            <h1 class="display-4 mb-4">GHAMECC Job Portal</h1>
            <p class="lead mb-5">Connect with talented members of the GHAMECC community for job opportunities and showcase your skills.</p>
            <div class="row justify-content-center">
                <div class="col-md-8">
                    <form action="talent-showcase.php" method="get" class="d-flex">
                        <input type="text" name="search" class="form-control form-control-lg me-2" placeholder="Search for skills, talents, or jobs...">
                        <button type="submit" class="btn btn-primary btn-lg">Search</button>
                    </form>
                </div>
            </div>
            <div class="mt-4">
                <?php if (isLoggedIn()): ?>
                    <a href="post-job.php" class="btn btn-outline-light me-2">Post a Job</a>
                    <a href="create-profile.php" class="btn btn-outline-light">Create Profile</a>
                <?php else: ?>
                    <a href="../login.php" class="btn btn-outline-light">Login to Post Jobs or Create Profile</a>
                <?php endif; ?>
            </div>
        </div>
    </section>
    
    <div class="container mb-5">
        <!-- Latest Jobs Section -->
        <section class="mb-5">
            <h2 class="section-heading">Latest Job Opportunities</h2>
            <div class="row">
                <?php if (empty($latestJobs)): ?>
                    <div class="col-12">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i> No job listings available at the moment.
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($latestJobs as $job): ?>
                        <div class="col-md-6 col-lg-4 mb-4">
                            <div class="card card-job h-100">
                                <div class="card-body">
                                    <span class="badge bg-primary mb-2"><?php echo htmlspecialchars($job['category_name'] ?? 'Uncategorized'); ?></span>
                                    <h5 class="card-title"><?php echo htmlspecialchars($job['title']); ?></h5>
                                    <p class="card-text text-muted mb-2">
                                        <i class="fas fa-map-marker-alt me-1"></i> <?php echo htmlspecialchars($job['location']); ?>
                                    </p>
                                    <p class="card-text mb-3"><?php echo substr(cleanText($job['description'], false), 0, 100) . '...'; ?></p>
                                    <div class="d-flex justify-content-between align-items-center">
                                        <small class="text-muted">
                                            <i class="far fa-clock me-1"></i> 
                                            <?php echo date('M d, Y', strtotime($job['created_at'])); ?>
                                        </small>
                                        <?php if (!empty($job['salary_range'])): ?>
                                            <small class="text-success">
                                                <i class="fas fa-money-bill-wave me-1"></i> 
                                                <?php echo htmlspecialchars($job['salary_range']); ?>
                                            </small>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="card-footer bg-white border-top-0">
                                    <a href="job-details.php?id=<?php echo $job['id']; ?>" class="btn btn-outline-primary btn-sm">View Details</a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    <div class="col-12 text-center mt-3">
                        <a href="job-listings.php" class="btn btn-outline-primary">View All Jobs</a>
                    </div>
                <?php endif; ?>
            </div>
        </section>
        
        <!-- Featured Talent Section -->
        <section class="mb-5">
            <h2 class="section-heading">Featured Talent</h2>
            <div class="row">
                <?php if (empty($featuredProfiles)): ?>
                    <div class="col-12">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i> No talent profiles available at the moment.
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($featuredProfiles as $profile): ?>
                        <div class="col-md-6 col-lg-3 mb-4">
                            <div class="card profile-card h-100">
                                <?php if (!empty($profile['profile_image'])): ?>
                                    <img src="../uploads/profiles/<?php echo $profile['profile_image']; ?>" class="card-img-top" alt="Profile Image">
                                <?php else: ?>
                                    <div class="bg-light text-center py-5">
                                        <i class="fas fa-user fa-4x text-secondary"></i>
                                    </div>
                                <?php endif; ?>
                                <div class="card-body">
                                    <h5 class="card-title"><?php echo htmlspecialchars($profile['first_name'] . ' ' . $profile['last_name']); ?></h5>
                                    <p class="card-text text-muted mb-2">
                                        <?php echo htmlspecialchars($profile['skills']); ?>
                                    </p>
                                    <div class="mb-2">
                                        <div class="star-rating">
                                            <?php 
                                            $avgRating = round($profile['avg_rating'] ?? 0);
                                            for ($i = 1; $i <= 5; $i++) {
                                                if ($i <= $avgRating) {
                                                    echo '<i class="fas fa-star"></i>';
                                                } else {
                                                    echo '<i class="far fa-star"></i>';
                                                }
                                            }
                                            ?>
                                            <small class="text-muted ms-1">(<?php echo $profile['rating_count'] ?? 0; ?>)</small>
                                        </div>
                                    </div>
                                    <p class="card-text">
                                        <span class="badge bg-<?php 
                                        switch($profile['availability']) {
                                            case 'full_time': echo 'success'; break;
                                            case 'part_time': echo 'info'; break;
                                            case 'contract': echo 'warning'; break;
                                            case 'freelance': echo 'primary'; break;
                                            default: echo 'secondary';
                                        }
                                        ?>">
                                            <?php echo ucwords(str_replace('_', ' ', $profile['availability'])); ?>
                                        </span>
                                    </p>
                                </div>
                                <div class="card-footer bg-white border-top-0">
                                    <a href="profile-details.php?id=<?php echo $profile['id']; ?>" class="btn btn-outline-primary btn-sm">View Profile</a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    <div class="col-12 text-center mt-3">
                        <a href="talent-showcase.php" class="btn btn-outline-primary">View All Talent</a>
                    </div>
                <?php endif; ?>
            </div>
        </section>
        
        <!-- Job Categories Section -->
        <section class="mb-5">
            <h2 class="section-heading">Browse Job Categories</h2>
            <div class="row">
                <?php if (empty($categories)): ?>
                    <div class="col-12">
                        <div class="alert alert-info">
                            <i class="fas fa-info-circle me-2"></i> No job categories available at the moment.
                        </div>
                    </div>
                <?php else: ?>
                    <?php foreach ($categories as $category): ?>
                        <div class="col-md-6 col-lg-3 mb-4">
                            <a href="job-listings.php?category=<?php echo $category['id']; ?>" class="text-decoration-none">
                                <div class="card category-card h-100">
                                    <div class="card-body">
                                        <div class="d-flex justify-content-between align-items-center">
                                            <h5 class="card-title mb-0"><?php echo htmlspecialchars($category['name']); ?></h5>
                                            <span class="badge bg-secondary"><?php echo $category['job_count']; ?></span>
                                        </div>
                                        <p class="card-text text-muted mt-2">
                                            <?php echo substr(htmlspecialchars($category['description']), 0, 80) . '...'; ?>
                                        </p>
                                    </div>
                                </div>
                            </a>
                        </div>
                    <?php endforeach; ?>
                    <div class="col-12 text-center mt-3">
                        <a href="categories.php" class="btn btn-outline-primary">View All Categories</a>
                    </div>
                <?php endif; ?>
            </div>
        </section>
        
        <!-- Call to Action Section -->
        <section class="text-center py-5 bg-light rounded">
            <div class="container">
                <h2 class="mb-4">Ready to Get Started?</h2>
                <p class="lead mb-4">Join our community of talented individuals and organizations looking for the perfect match.</p>
                <div class="row justify-content-center">
                    <div class="col-md-8">
                        <div class="d-flex flex-column flex-md-row justify-content-center gap-3">
                            <?php if (isLoggedIn()): ?>
                                <a href="post-job.php" class="btn btn-primary">Post a Job</a>
                                <a href="create-profile.php" class="btn btn-success">Create Your Profile</a>
                                <a href="talent-showcase.php" class="btn btn-info text-white">Find Talent</a>
                            <?php else: ?>
                                <a href="../login.php" class="btn btn-primary">Login</a>
                                <a href="../login.php?action=register" class="btn btn-success">Register</a>
                                <a href="talent-showcase.php" class="btn btn-info text-white">Browse Talent</a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </section>
    </div>
    
    <!-- Footer -->
    <footer class="bg-dark text-white py-4">
        <div class="container">
            <div class="row">
                <div class="col-md-4 mb-4 mb-md-0">
                    <h5>GHAMECC Job Portal</h5>
                    <p class="text-muted">Connecting talent with opportunities within the GHAMECC community.</p>
                </div>
                <div class="col-md-4 mb-4 mb-md-0">
                    <h5>Quick Links</h5>
                    <ul class="list-unstyled">
                        <li><a href="job-listings.php" class="text-decoration-none text-muted">Jobs</a></li>
                        <li><a href="talent-showcase.php" class="text-decoration-none text-muted">Talent Showcase</a></li>
                        <li><a href="categories.php" class="text-decoration-none text-muted">Categories</a></li>
                        <?php if (isLoggedIn()): ?>
                            <li><a href="create-profile.php" class="text-decoration-none text-muted">Create Profile</a></li>
                        <?php else: ?>
                            <li><a href="../login.php" class="text-decoration-none text-muted">Login</a></li>
                        <?php endif; ?>
                    </ul>
                </div>
                <div class="col-md-4">
                    <h5>Contact</h5>
                    <address class="text-muted">
                        <i class="fas fa-map-marker-alt me-2"></i> GHAMECC Headquarters<br>
                        <i class="fas fa-phone me-2"></i> (123) 456-7890<br>
                        <i class="fas fa-envelope me-2"></i> jobs@ghamecc.org
                    </address>
                </div>
            </div>
            <hr class="my-4 bg-secondary">
            <div class="text-center text-muted">
                <small>&copy; <?php echo date('Y'); ?> GHAMECC Job Portal. All rights reserved.</small>
            </div>
        </div>
    </footer>
    
    <!-- Bootstrap JS Bundle with Popper -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
