<?php
/**
 * Job Portal - Categories Page
 * GHAMECC Choir Management Platform
 * 
 * This page displays job categories and allows users to browse jobs by category.
 */

// Include database connection and functions
require_once '../config/database.php';
require_once '../includes/functions.php';

// Get all job categories
$categories = fetchRows("SELECT c.*, COUNT(j.id) as job_count 
                        FROM job_categories c 
                        LEFT JOIN jobs j ON c.id = j.category_id AND j.status = 'open' AND j.approval_status = 'approved'
                        GROUP BY c.id 
                        ORDER BY c.name ASC");

// Get featured jobs (latest 3 jobs)
$featuredJobs = fetchRows("SELECT j.*, c.name as category_name 
                          FROM jobs j 
                          LEFT JOIN job_categories c ON j.category_id = c.id 
                          WHERE j.status = 'open' AND j.approval_status = 'approved'
                          ORDER BY j.created_at DESC 
                          LIMIT 3");

// Page title
$pageTitle = "Job Categories - GHAMECC Job Portal";
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $pageTitle; ?></title>
    
    <?php include '../includes/favicon.php'; ?>
    
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .category-card {
            transition: transform 0.3s, box-shadow 0.3s;
            height: 100%;
        }
        .category-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        .category-icon {
            font-size: 2.5rem;
            margin-bottom: 15px;
            color: #0d6efd;
        }
        .job-count {
            background-color: #0d6efd;
            color: white;
            border-radius: 20px;
            padding: 3px 10px;
            font-size: 0.8rem;
            display: inline-block;
        }
        .featured-job {
            border-left: 4px solid #0d6efd;
            transition: transform 0.3s;
        }
        .featured-job:hover {
            transform: translateX(5px);
        }
    </style>
</head>
<body>
    <!-- Navigation -->
    <?php include 'includes/job-portal-navbar.php'; ?>

    <!-- Main Content -->
    <div class="container py-5">
        <div class="row mb-4">
            <div class="col-md-8">
                <h1 class="mb-3">Job Categories</h1>
                <p class="lead">Browse jobs by category to find opportunities that match your skills and interests.</p>
            </div>
            <div class="col-md-4 text-md-end">
                <a href="index.php" class="btn btn-outline-primary">
                    <i class="fas fa-list"></i> View All Jobs
                </a>
            </div>
        </div>

        <!-- Categories Section -->
        <div class="row row-cols-1 row-cols-md-2 row-cols-lg-3 g-4 mb-5">
            <?php if (empty($categories)): ?>
                <div class="col-12">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> No job categories found.
                    </div>
                </div>
            <?php else: ?>
                <?php foreach ($categories as $category): ?>
                    <div class="col">
                        <div class="card category-card h-100">
                            <div class="card-body text-center">
                                <div class="category-icon">
                                    <i class="fas fa-briefcase"></i>
                                </div>
                                <h3 class="card-title"><?php echo htmlspecialchars($category['name']); ?></h3>
                                <span class="job-count"><?php echo $category['job_count']; ?> Jobs</span>
                                <p class="card-text mt-3">
                                    <?php echo !empty($category['description']) ? 
                                        htmlspecialchars(truncateText($category['description'], 100)) : 
                                        'Find jobs in this category'; ?>
                                </p>
                                <a href="index.php?category=<?php echo $category['id']; ?>" class="btn btn-primary mt-2">
                                    Browse Jobs
                                </a>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <!-- Featured Jobs Section -->
        <div class="row mt-5">
            <div class="col-12">
                <h2 class="mb-4">Featured Jobs</h2>
            </div>
            
            <?php if (empty($featuredJobs)): ?>
                <div class="col-12">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> No featured jobs available at the moment.
                    </div>
                </div>
            <?php else: ?>
                <div class="col-lg-8">
                    <?php foreach ($featuredJobs as $job): ?>
                        <div class="card mb-3 featured-job">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-center mb-2">
                                    <h5 class="card-title mb-0">
                                        <a href="job-details.php?id=<?php echo $job['id']; ?>" class="text-decoration-none">
                                            <?php echo htmlspecialchars($job['title']); ?>
                                        </a>
                                    </h5>
                                    <span class="badge bg-primary"><?php echo htmlspecialchars($job['category_name']); ?></span>
                                </div>
                                <h6 class="card-subtitle mb-2 text-muted">
                                    <?php echo htmlspecialchars($job['company_name']); ?> • 
                                    <?php echo htmlspecialchars($job['location']); ?>
                                </h6>
                                <p class="card-text">
                                    <?php echo htmlspecialchars(truncateText(strip_tags($job['description']), 150)); ?>
                                </p>
                                <div class="d-flex justify-content-between align-items-center">
                                    <small class="text-muted">
                                        <i class="far fa-clock"></i> 
                                        Posted <?php echo timeElapsed($job['created_at']); ?>
                                    </small>
                                    <a href="job-details.php?id=<?php echo $job['id']; ?>" class="btn btn-sm btn-outline-primary">
                                        View Details
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                    <div class="text-center mt-4">
                        <a href="index.php" class="btn btn-primary">View All Jobs</a>
                    </div>
                </div>
                
                <div class="col-lg-4">
                    <div class="card">
                        <div class="card-header bg-primary text-white">
                            <h5 class="mb-0">Looking for talent?</h5>
                        </div>
                        <div class="card-body">
                            <p>Post a job and reach qualified GHAMECC members with the skills you need.</p>
                            <div class="d-grid">
                                <a href="post-job.php" class="btn btn-primary">Post a Job</a>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card mt-4">
                        <div class="card-header bg-secondary text-white">
                            <h5 class="mb-0">Showcase Your Skills</h5>
                        </div>
                        <div class="card-body">
                            <p>Create a profile to showcase your skills and get discovered by potential employers.</p>
                            <div class="d-grid">
                                <a href="talent-showcase.php" class="btn btn-secondary">Join Talent Showcase</a>
                            </div>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white py-4 mt-5">
        <div class="container">
            <div class="row">
                <div class="col-md-4 mb-3">
                    <h5>GHAMECC Job Portal</h5>
                    <p>A platform for Ghana Methodist Church Choir members to find and post job opportunities.</p>
                </div>
                <div class="col-md-4 mb-3">
                    <h5>Quick Links</h5>
                    <ul class="list-unstyled">
                        <li><a href="index.php" class="text-white">Home</a></li>
                        <li><a href="post-job.php" class="text-white">Post a Job</a></li>
                        <li><a href="talent-showcase.php" class="text-white">Talent Showcase</a></li>
                        <li><a href="about.php" class="text-white">About</a></li>
                    </ul>
                </div>
                <div class="col-md-4 mb-3">
                    <h5>Contact</h5>
                    <address>
                        <p><i class="fas fa-envelope me-2"></i> info@ghamecc.org</p>
                        <p><i class="fas fa-phone me-2"></i> +233 XX XXX XXXX</p>
                    </address>
                </div>
            </div>
            <hr>
            <div class="text-center">
                <p>&copy; <?php echo date('Y'); ?> GHAMECC Job Portal. All rights reserved.</p>
            </div>
        </div>
    </footer>

    <!-- Bootstrap JS -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
