<?php
/**
 * Custom Logger Utility
 * GHAMECC Choir Management Platform
 */

// Define log levels
define('LOG_LEVEL_INFO', 'INFO');
define('LOG_LEVEL_WARNING', 'WARNING');
define('LOG_LEVEL_ERROR', 'ERROR');
define('LOG_LEVEL_DEBUG', 'DEBUG');

// Define log file paths
define('LOG_FILE_PATH', __DIR__ . '/../logs/app.log');
define('ERROR_LOG_FILE_PATH', __DIR__ . '/../logs/error.log');
define('MEMBERS_LOG_FILE_PATH', __DIR__ . '/../logs/members.log');

// Make sure logs directory exists
if (!file_exists(__DIR__ . '/../logs')) {
    mkdir(__DIR__ . '/../logs', 0755, true);
}

// Ensure log files exist and PHP error logging targets the dedicated error log
foreach ([LOG_FILE_PATH, ERROR_LOG_FILE_PATH, MEMBERS_LOG_FILE_PATH] as $logFilePath) {
    if (!file_exists($logFilePath)) {
        touch($logFilePath);
    }
}

ini_set('log_errors', '1');
ini_set('error_log', ERROR_LOG_FILE_PATH);

/**
 * Log a message to the application log file
 * 
 * @param string $message The message to log
 * @param string $level The log level (INFO, WARNING, ERROR, DEBUG)
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function appLog($message, $level = LOG_LEVEL_INFO, $context = []) {
    // Format the log entry
    $timestamp = date('Y-m-d H:i:s');
    $user = isset($_SESSION['username']) ? $_SESSION['username'] : 'system';
    $userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 'none';
    $role = isset($_SESSION['role']) ? $_SESSION['role'] : 'none';
    
    // Format context data as JSON if not empty
    $contextString = !empty($context) ? ' | Context: ' . json_encode($context) : '';
    
    // Create log entry
    $logEntry = "[$timestamp] [$level] [$user/$userId/$role] $message$contextString" . PHP_EOL;
    
    // Write to log file
    $result = file_put_contents(LOG_FILE_PATH, $logEntry, FILE_APPEND | LOCK_EX);

    if ($level === LOG_LEVEL_ERROR) {
        file_put_contents(ERROR_LOG_FILE_PATH, $logEntry, FILE_APPEND | LOCK_EX);
    }
    
    return ($result !== false);
}

/**
 * Log a message to the members log file
 * 
 * @param string $message The message to log
 * @param string $level The log level (INFO, WARNING, ERROR, DEBUG)
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function memberLog($message, $level = LOG_LEVEL_INFO, $context = []) {
    // Format the log entry
    $timestamp = date('Y-m-d H:i:s');
    $user = isset($_SESSION['username']) ? $_SESSION['username'] : 'system';
    $userId = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 'none';
    $role = isset($_SESSION['role']) ? $_SESSION['role'] : 'none';
    
    // Format context data as JSON if not empty
    $contextString = !empty($context) ? ' | Context: ' . json_encode($context) : '';
    
    // Create log entry
    $logEntry = "[$timestamp] [$level] [$user/$userId/$role] $message$contextString" . PHP_EOL;
    
    // Write to members log file
    $result = file_put_contents(MEMBERS_LOG_FILE_PATH, $logEntry, FILE_APPEND | LOCK_EX);
    
    // Also write to main app log for critical levels
    if ($level == LOG_LEVEL_ERROR || $level == LOG_LEVEL_WARNING) {
        $prefixedEntry = "[MEMBER] " . $logEntry;
        file_put_contents(LOG_FILE_PATH, $prefixedEntry, FILE_APPEND | LOCK_EX);
        file_put_contents(ERROR_LOG_FILE_PATH, $prefixedEntry, FILE_APPEND | LOCK_EX);
    }
    
    return ($result !== false);
}

/**
 * Log an informational message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logInfo($message, $context = []) {
    return appLog($message, LOG_LEVEL_INFO, $context);
}

/**
 * Log a warning message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logWarning($message, $context = []) {
    return appLog($message, LOG_LEVEL_WARNING, $context);
}

/**
 * Log an error message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logError($message, $context = []) {
    return appLog($message, LOG_LEVEL_ERROR, $context);
}

/**
 * Log a debug message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logDebug($message, $context = []) {
    return appLog($message, LOG_LEVEL_DEBUG, $context);
}

/**
 * Log member-related informational message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logMemberInfo($message, $context = []) {
    return memberLog($message, LOG_LEVEL_INFO, $context);
}

/**
 * Log member-related warning message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logMemberWarning($message, $context = []) {
    return memberLog($message, LOG_LEVEL_WARNING, $context);
}

/**
 * Log member-related error message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logMemberError($message, $context = []) {
    return memberLog($message, LOG_LEVEL_ERROR, $context);
}

/**
 * Log member-related debug message
 * 
 * @param string $message The message to log
 * @param array $context Additional contextual data
 * @return bool True if log was written successfully, false otherwise
 */
function logMemberDebug($message, $context = []) {
    return memberLog($message, LOG_LEVEL_DEBUG, $context);
}
